# coding: utf-8
# python 2 only

###########################################################
##                                                       ##
##           TOOL BREAKAGE NOTIFIER v0.95                ##
##           https://www.tormachtips.com                 ##
##                                                       ##
###########################################################

"""
DESCRIPTION:
    This PathPilot plugin monitors for important machine errors,
    such as tool breakage or ETS mismatches, by intercepting messages
    written to the internal error handler. When a watched phrase is detected,
    an email alert is automatically sent to a configured recipient.

    It’s commonly used with G37 — an ETS command that verifies
    tool length against expected values. Deviations trigger a soft alarm,
    which this plugin can detect and notify you about.
    
    Use G37 P0.005 (for example). This checks if the currently loaded tool is 
    within 0.005" of tolerance when the ETS is tripped. If it's too long (wrong tool) 
    or too short (wrong tool or broken tool), it will alert PathPilot and 
    thus this script and will email you.

INSTALLATION:
    1. Copy both of the following files into your PathPilot system:
          - breakage_plugin.py (this script)
          - breakage_config.ini (configuration file)

    2. Place both files into:
          /home/operator/gcode/python/

    3. Ensure this required support file is also present in that folder:
          - ui_hooks.py

    4. Edit the config (.ini) file. The fields are pretty straightforward.

    Notes:
    - `\\n` in the body becomes a newline.
    - `{msg}` is replaced with the raw PathPilot error message.
    - `{match}` is replaced with the string from [strings] that triggered the alert.
    - You may leave unused `string#` entries blank — they’ll be ignored.

NOTES:
    - The plugin hot-loads the INI on every event trigger.
    - No restart of PathPilot is needed to apply config changes.
    - Email sending is done in the background and does not interfere with machine performance.
"""

import threading
import time
import types
import os
import smtplib
import ConfigParser
from email.mime.text import MIMEText
from ui_hooks import plugin
import singletons
import constants

class UserPlugin(plugin):
    def __init__(self):
        plugin.__init__(self, "Tool Break Detector")
        self.config_file = os.path.expanduser('~/gcode/python/breakage_config.ini')
        t = threading.Thread(target=self.setup_hook)
        t.daemon = True
        t.start()
        self.ShowMsg("Tool breakage watcher loaded (via error_handler.write)", constants.ALARM_LEVEL_QUIET)

    def setup_hook(self):
        while True:
            try:
                if hasattr(singletons, 'g_Machine') and singletons.g_Machine:
                    self.ui = singletons.g_Machine
                    err_handler = getattr(self.ui, 'error_handler', None)
                    if not err_handler:
                        raise Exception("No error_handler found")
                    original_func = err_handler.write
                    def make_wrapper(plugin_self, orig_func):
                        def wrapped(this, msg, level=constants.ALARM_LEVEL_QUIET, **kwargs):
                            if isinstance(msg, basestring):
                                msg_lower = msg.lower()
                                phrases = plugin_self._load_watch_phrases()
                                for phrase in phrases:
                                    if phrase in msg_lower:
                                        plugin_self.on_tool_break(msg, matched_phrase=phrase)
                                        break
                            return orig_func(msg, level, **kwargs)
                        return wrapped
                    err_handler.write = types.MethodType(make_wrapper(self, original_func), err_handler)
                    self.ShowMsg("Hook on error_handler.write applied", constants.ALARM_LEVEL_QUIET)
                    return
            except Exception as e:
                self.ShowMsg("Hook error: " + str(e), constants.ALARM_LEVEL_LOW)
            time.sleep(0.5)

    def _load_watch_phrases(self):
        phrases = []
        try:
            if not os.path.exists(self.config_file):
                return phrases
            config = ConfigParser.ConfigParser()
            config.read(self.config_file)
            if config.has_section('strings'):
                for key, val in config.items('strings'):
                    val = val.strip()
                    if val:
                        phrases.append(val.lower())
        except Exception as e:
            self.ShowMsg("Failed to load phrases: " + str(e), constants.ALARM_LEVEL_LOW)
        return phrases   
    
    def send_tool_break_email(self, msg, matched_phrase=None):
        try:
            if not os.path.exists(self.config_file):
                raise IOError("Config file not found")
            config = ConfigParser.ConfigParser()
            config.read(self.config_file)
            sender = config.get('email', 'sender')
            recipient = config.get('email', 'recipient')
            subject = config.get('email', 'subject') or "Tool Break Detected"         
            body_template = config.get('email', 'body') or "Tool break detected.\n\nDetails:\n{msg}"
            body = body_template.replace('\\n', '\n').format(msg=msg, match=matched_phrase)                       
            smtp_server = config.get('email', 'smtp_server')
            smtp_port = config.getint('email', 'smtp_port')
            smtp_username = config.get('email', 'smtp_username')
            smtp_password = config.get('email', 'smtp_password')
            mime_msg = MIMEText(body)
            mime_msg['From'] = sender
            mime_msg['To'] = recipient
            mime_msg['Subject'] = subject
            smtp = smtplib.SMTP_SSL(smtp_server, smtp_port)
            smtp.login(smtp_username, smtp_password)
            smtp.sendmail(sender, [recipient], mime_msg.as_string())
            smtp.quit()
            self.ShowMsg("Tool break alert email sent", constants.ALARM_LEVEL_MEDIUM)
        except Exception as e:
            self.ShowMsg("Tool break email failed: " + str(e), constants.ALARM_LEVEL_LOW)

    def is_enabled(self):
        try:
            if not os.path.exists(self.config_file):
                return False
            config = ConfigParser.ConfigParser()
            config.read(self.config_file)
            return config.getint('settings', 'enabled') == 1
        except:
            return False

    def on_tool_break(self, msg, matched_phrase=None):
        if not self.is_enabled():
            return
        self.ShowMsg("TOOL BREAK DETECTED", constants.ALARM_LEVEL_HIGH)
        print "[Tool Break Detected] %s" % msg
        t = threading.Thread(target=self.send_tool_break_email, args=(msg, matched_phrase))
        t.daemon = True
        t.start()